<?php namespace App\Controllers\Api\Customer\Payment\PayPal;

use App\Controllers\PrivateController;
use App\Libraries\Settings;
use App\Models\AppsModel;
use App\Models\DepositMethodsModel;
use App\Models\PlansModel;
use App\Models\SubscribesModel;
use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\Database\Exceptions\DatabaseException;

class CreatePaymentRequest extends PrivateController
{
    public function index(): ResponseInterface
    {
        if (!$this->validate($this->create_validation_type())) {
            return $this->respond(["message" => $this->validator->getErrors()], 400);
        }

        $uid = esc($this->request->getGet("uid"));

        $projects = new AppsModel();
        $db = \Config\Database::connect();
        $db->transStart();

        try {
            $app = $projects->where([
                "uid" => $uid,
                "user" => $this->userId,
                "deleted_at" => 0
            ])->select("id")->first();

            if (!$app) {
                throw new DatabaseException(lang("Message.message_14"));
            }

            $subscribes = new SubscribesModel();
            $is_active = $subscribes->where([
                "app_id" => $app["id"],
                "expires_at >" => time(),
                "is_disable" => 0
            ])->countAllResults();

            if ($is_active) {
                throw new DatabaseException(lang("Message.message_86"));
            }

            $plans = new PlansModel();
            $plan = $plans->where([
                "id" => (int)$this->request->getJsonVar("plan_id"),
                "status" => 1,
                "deleted_at" => 0
            ])->first();

            if (!$plan) {
                throw new DatabaseException(lang("Message.message_83"));
            }

            $methods = new DepositMethodsModel();
            $method = $methods->where([
                "name" => "PayPal",
                "status" => 1
            ])->first();

            if (!$method) {
                throw new DatabaseException(lang("Message.message_84"));
            }

            $settings = new Settings();
            $frontUrl = rtrim($settings->get_config("site_url"), '/') . '/';

            // Sử dụng base_url() để lấy giá trị baseURL từ cấu hình và giữ lại như phiên bản cũ
            $returnUrl = base_url('public/ipn/paypal/capture');
            $cancelUrl = $frontUrl . 'private/profile/subscribe';

            $paypalUrl = 'https://api.paypal.com/v2/checkout/orders'; // Chuyển sang môi trường live

            $clientId = $method["api_value_1"];
            $clientSecret = $method["api_value_2"];
            $auth = base64_encode("$clientId:$clientSecret");

            // In ra mã xác thực để kiểm tra
            log_message('debug', 'PayPal Auth: ' . $auth);

            $client = service('curlrequest');
            $response = $client->post($paypalUrl, [
                'headers' => [
                    'Authorization' => "Basic $auth",
                    'Content-Type' => 'application/json'
                ],
                'json' => [
                    'intent' => 'CAPTURE',
                    'purchase_units' => [[
                        'amount' => [
                            'currency_code' => 'USD',
                            'value' => $plan["price"]
                        ],
                        'custom_id' => $plan["id"]."_" .  $this->userId."_" . $app["id"],
                    ]],
                    'application_context' => [
                        'return_url' => $returnUrl, // Sử dụng base_url() từ cấu hình
                        'cancel_url' => $cancelUrl // Giữ nguyên như phiên bản cũ
                    ]
                ]
            ]);
            $order = json_decode($response->getBody(), true);

            if (isset($order['links'][1]['href'])) {
                $db->transComplete();
                return $this->respond(["url" => $order['links'][1]['href']], 200);
            } else {
                throw new DatabaseException("PayPal order creation failed");
            }
        } catch (\Exception $e) {
            $db->transRollback();
            return $this->respond(["message" => $e->getMessage()], 400);
        }
    }

    private function create_validation_type(): array
    {
        return [
            "plan_id" => [
                "label" => lang("Fields.field_148"),
                "rules" => "required|numeric"
            ],
        ];
    }
}